unit aspxDelphiWCL.ColorSelector;

{
Copyright:  aspxDelphi.net All rights reserved. Use freely.

Zarko Gajic
http://aspxdelphi.net
http://delphi.about.com

}

interface

uses
  aspxDelphiWCL.Common, System.Web, System.Web.UI, System.Web.UI.WebControls, System.Drawing,
  System.ComponentModel, System.Collections.Specialized;

type
  SelectedColorChangedEvent = procedure(e : System.EventArgs) of object;

  /// <summary>
  /// A drop down pick list that lets users select a Color value
  /// </summary>
  [DefaultProperty('SelectedColor')
	,DefaultEvent('SelectedColorChanged')
  ,ToolboxData('<{0}:ColorSelector runat=server></{0}:ColorSelector>')
  ]
  ColorSelector = class(System.Web.UI.WebControls.WebControl, INamingContainer, IPostBackDataHandler)
  strict private
    type TColor = record
      Color : System.Drawing.Color;
      Selected : boolean;
    end;
  strict private
    type TColorArray = array of TColor;
  strict private
    FSelectedColorChangedEvent : SelectedColorChangedEvent;
    FSelectedColor: System.Drawing.Color;
    Colors : TColorArray;
  strict protected
    procedure BindColors;
    procedure RenderContents(writer: HtmlTextWriter); override;
    function CreateControlCollection : ControlCollection; override;
    procedure OnInit(e: EventArgs); override;
    procedure OnPreRender(e:EventArgs); override;
    procedure AddAttributesToRender(writer : HtmlTextWriter); override;
  public
    constructor Create;

    function LoadPostData(postDataKey: string;  postCollection: NameValueCollection): boolean;
    procedure RaisePostDataChangedEvent;
  protected
    procedure OnSelectedColorChanged;
  published
    function getAutoPostback: boolean;
    procedure setAutopostback(const Value: boolean);
    procedure setColor(const Value: System.Drawing.Color);

    [Category('Appearance')	,DefaultValue(typeof(Color), 'Empty')]
    property SelectedColor: System.Drawing.Color  read FSelectedColor write setColor;

    [Category('Behavior'), DefaultValue(false)]
    property AutoPostBack : boolean read getAutoPostback write setAutopostback;

    //events
    property SelectedColorChanged : SelectedColorChangedEvent add FSelectedColorChangedEvent remove FSelectedColorChangedEvent;
  end;

implementation

{$REGION 'IPostBackDataHandler'}
function ColorSelector.LoadPostData(postDataKey: string; postCollection: NameValueCollection): boolean;
var
  clr : Color;
begin
  clr := ColorTranslator.FromHtml(postCollection[postDataKey]);
  if clr <> SelectedColor then //Call the RaisePostDataChangedEvent
  begin
    SelectedColor := clr;
    Result := true;
  end
  else
    Result := false;
end;

procedure ColorSelector.RaisePostDataChangedEvent;
begin
  OnSelectedColorChanged;
end;

{$ENDREGION}

procedure ColorSelector.BindColors;
type
  TStringArray = array of string;
var
  clrs : TStringArray;
  i :  integer;
begin
  clrs := System.Enum.GetNames(typeof(System.Drawing.KnownColor));
  for i:= 0 to -1 + System.&Array(Colors).Length do
  begin
    TColor(Colors[i]).Color := Color.FromName(clrs[i]);
    TColor(Colors[i]).Selected := false;
  end;
end;

constructor ColorSelector.Create;
begin
  inherited Create(HtmlTextWriterTag.Select);
  SetLength(Colors,System.Array(System.Enum.GetNames(typeof(System.Drawing.KnownColor))).Length);

  FSelectedColor := Color.Empty;

  BindColors;

end;

function ColorSelector.CreateControlCollection: ControlCollection;
begin
  result := EmptyControlCollection.Create(Self);
end;

function ColorSelector.getAutoPostback: boolean;
var
  o: System.Object;
begin
  result:= false;
  o := ViewState['AutoPostBack'];
  if (o <> nil) then
  begin
    result := Boolean(o);
  end
end;

procedure ColorSelector.OnInit(e: EventArgs);
begin
  inherited;

  if not Common.IsDesignTime(Page) then
    Page.RegisterRequiresPostBack(self);
end;

procedure ColorSelector.RenderContents(writer: HtmlTextWriter);
var
  coll: TColorArray;
  cnt : Integer;
  multi: Boolean;
  idx: Integer;
  item: TColor;

  colorName : string;
  colorValue : integer;
begin
  coll := Colors;
  cnt := System.Array(Colors).Length;
  multi := false;
  if (cnt > 0) then
  begin
    for idx := 0 to cnt-1 do
    begin
      item := TColor(coll[idx]);

      //don't use system colors
      if item.Color.IsSystemColor then Continue;

      colorName := item.Color.Name;
      colorValue := item.Color.ToArgb();

      writer.WriteBeginTag('option');
      if item.Selected then
      begin
        if multi then raise HttpException.Create('Cannot_Multiselect_In_ColorDropDownList');

        multi := true;
        writer.WriteAttribute('selected', 'selected');
      end;
      writer.WriteAttribute('value', ColorTranslator.ToHtml(item.Color));

      writer.Write(' style="');
      writer.WriteStyleAttribute('background-color',Common.ColorToHex(Color.FromArgb(colorValue)));
      writer.Write('"');

      writer.Write(HtmlTextWriter.TagRightChar);
      HttpUtility.HtmlEncode(colorName, writer);
      writer.WriteEndTag('option');
      writer.WriteLine;
    end;
  end;
end;

procedure ColorSelector.setAutopostback(const Value: boolean);
begin
  ViewState['AutoPostBack'] := System.Object(value);
end;

procedure ColorSelector.setColor(const Value: System.Drawing.Color);
var
  i : integer;
begin
  if FSelectedColor <> Value then
  begin
    FSelectedColor := Value;

    for i := 0 to -1 + Length(Colors) do
    begin
      TColor(Colors[i]).Selected := False;
      if Equals(TColor(Colors[i]).Color, Value) then
      begin
        TColor(Colors[i]).Selected := True;
      end;
    end;
  end;
end;

procedure ColorSelector.OnSelectedColorChanged;
begin
  if Assigned(FSelectedColorChangedEvent) then
    FSelectedColorChangedEvent(EventArgs.Empty);

end;

procedure ColorSelector.OnPreRender(e: EventArgs);
begin
  inherited OnPreRender(e);


end;

procedure ColorSelector.AddAttributesToRender(writer: HtmlTextWriter);
begin
  inherited;
  //must be inside form tags
  if not Common.IsDesignTime(Page) then Page.VerifyRenderingInServerForm(Self);

  //autopostback?
  if (AutoPostBack) then writer.AddAttribute('onchange', 'javascript:' + Page.GetPostBackEventReference(Self, ''));

  //add name attrubute
  writer.AddAttribute('name', inherited UniqueID);
end;

end.
